<?php

namespace Fir\Controllers;

use Fir\Models;
use Fir\Views;
use Fir\Languages\Language as Language;

/**
 * The base Controller upon which all the other controllers are extended on
 */
class Controller {

    /**
     * The database connection
     * @var	object
     */
    public $db;

    /**
     * The site settings from the DB
     * @var	array
     */
    protected $settings;

    /**
     * The view object to be passed to the controllers
     * @var	object
     */
    protected $view;

    /**
     * The language array to be passed to the controllers and views
     * @var array
     */
    protected $lang;

    /**
     * The list of available languages
     * @var array
     */
    protected $languages;

    /**
     * User selected language
     * @var string
     */
    protected $language;

    /**
     * The current URL path (route) array to be passed to the controllers
     * @var array
     */
    protected $url;

    /**
     * Controller constructor.
     * @param $db
     */
    public function __construct($db, $url) {
        $this->db = $db;
        $this->url = $url;

        // Instantiate the main Model
        $model = new Models\Model($this->db);

        // Store the site settings
        $this->settings = $model->getSiteSettings();

        // Set the timezone
        if(!empty($this->settings['timezone'])) {
            date_default_timezone_set($this->settings['timezone']);
        }

        // Load default user settings
        $this->defaultUserSettings();

        // Instantiate the Language system and set the default language
        $language = new Language();

        $this->lang = $language->set($this->settings['site_language']);
        $this->languages = $language->languages;
        $this->language = $language->get();

        // Instantiate the View
        $this->view = new Views\View($this->settings, $this->lang, $this->url);
    }

    /**
     * Get and instantiate the requested model
     *
     * @return	object
     */
    public function model($model) {
        require_once(__DIR__ . '/../models/' . $model . '.php');

        /*
         * The namespace\class must be defined in a string as it can't be called shorted using new namespace\$var
         */
        $class = 'Fir\Models\\'.$model;

        return new $class($this->db);
    }

    /**
     * Output the final view to the user based on the request type
     *
     * @param   $data   array   The output generated by the controllers
     * @return	string
     */
    public function run($data = null) {

        if($this->url[0] == 'requests') {
            if(isAjax()) {
                echo json_encode($data);
            } else {
                redirect();
            }
        } else {
            $data['header_view'] = $this->getHeader();
            $data['content_view'] = $data['content'];
            $data['footer_view'] = $this->getFooter();
            $data['backgrounds'] = $this->getBackgrounds();
            if(isAjax()) {
                echo json_encode(['title' => $this->view->docTitle(), 'header' => $data['header_view'], 'content' => $data['content_view'], 'footer' => $data['footer_view']]);
            } else {
                echo $this->view->render($data, 'wrapper');
            }
        }
    }

    private function getHeader() {
        if($this->settings['web_per_page'] > 0) {
            $data['menu'][] = ['web', false];
        }
        if($this->settings['images_per_page'] > 0) {
            $data['menu'][] = ['images', false];
        }
        if($this->settings['videos_per_page'] > 0) {
            $data['menu'][] = ['videos', false];
        }
        if($this->settings['news_per_page'] > 0) {
            $data['menu'][] = ['news', false];
        }

        // Enable the default search page
        $data['menu'][0][1] = true;

        $data['site_menu'] = [
            'preferences' => [
                'language'  => [false],
                'theme'     => [false],
                'search'    => [false]
            ]
        ];

        $data['query'] = isset($_GET['q']) ? $_GET['q'] : '';
        if(isset($this->url[0]) && in_array($this->url[0], ['web', 'images', 'videos', 'news'])) {
            $data['top_bar'] = true;
            $data['query_path'] = $this->url[0];
            $data['autofocus'] = 0;
            $data['search_bar_view'] = $this->view->render($data, 'shared/search_bar');
        }
        return $this->view->render($data, 'shared/header');
    }

    private function getFooter() {
        $footer = $this->model('Wrapper');
        $data['info_pages'] = $footer->getInfoPages();
        return $this->view->render($data, 'shared/footer');
    }

    /**
     * The Search menu
     *
     * @return  array
     */
    protected function getMenu() {
        $data['query'] = $_GET['q'];

        if($this->settings['web_per_page'] > 0) {
            $data['menu']['web'] = [false];
        }
        if($this->settings['images_per_page'] > 0) {
            $data['menu']['images'] = [false];
        }
        if($this->settings['videos_per_page'] > 0) {
            $data['menu']['videos'] = [false];
        }
        if($this->settings['news_per_page'] > 0) {
            $data['menu']['news'] = [false];
        }

        // If on the current route, enable the Active flag
        $data['menu'][$this->url[0]][0] = true;

        return $this->view->render($data, 'shared/search_menu');
    }

    /**
     * Returns a list of available backgrounds
     *
     * @return  array
     */
    protected function getBackgrounds() {
        // Define the languages folder
        $this->folder = __DIR__ .'/../../public/uploads/backgrounds/';

        $backgrounds = [];

        if($handle = opendir($this->folder)) {
            while(false !== ($entry = readdir($handle))) {
                $pathInfo = pathinfo($entry);

                if($entry != '.' && $entry != '..' && in_array(strtolower($pathInfo['extension']), ['jpg', 'jpeg', 'png'])) {
                    $backgrounds[] = $pathInfo['basename'];
                }
            }
            closedir($handle);
        }

        return $backgrounds;
    }

    /**
     * Sets the backgrounds cookie if the settings allows it
     */
    protected function defaultUserSettings() {
        if(isset($_COOKIE['backgrounds']) == false) {
            setcookie("backgrounds", $this->settings['site_backgrounds'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['backgrounds'] = $this->settings['site_backgrounds'];
        }

        if(isset($_COOKIE['dark_mode']) == false) {
            setcookie("dark_mode", $this->settings['site_dark_mode'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['dark_mode'] = $this->settings['site_dark_mode'];
        }

        if(isset($_COOKIE['center_content']) == false) {
            setcookie("center_content", $this->settings['site_center_content'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['center_content'] = $this->settings['site_center_content'];
        }

        if(isset($_COOKIE['safe_search']) == false || in_array($_COOKIE['safe_search'], ['Off', 'Moderate', 'Strict']) == false) {
            setcookie("safe_search", $this->settings['search_safe_search'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['safe_search'] = $this->settings['search_safe_search'];
        }

        if(isset($_COOKIE['new_window']) == false) {
            setcookie("new_window", $this->settings['search_new_window'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['new_window'] = $this->settings['search_new_window'];
        }

        if(isset($_COOKIE['highlight']) == false || in_array($_COOKIE['highlight'], ['false', 'true']) == false) {
            setcookie("highlight", $this->settings['search_highlight'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['highlight'] = $this->settings['search_highlight'];
        }

        if(isset($_COOKIE['market']) == false) {
            setcookie("market", $this->settings['search_market'], time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['market'] = $this->settings['search_market'];
        }

        if(isset($_COOKIE['cookie_law']) == false) {
            setcookie("cookie_law", 0, time() + (10 * 365 * 24 * 60 * 60), COOKIE_PATH);
            $_COOKIE['cookie_law'] = 0;
        }

        return;
    }
}